<?php

namespace Tests\Feature;

use App\Enums\StatusEnum;
use App\Enums\TipoPessoaTomador;
use App\Models\Cidade;
use App\Models\DadosCadastrais;
use App\Models\Instituicao;
use App\Models\Lote;
use App\Models\Prestador;
use App\Models\Rps;
use App\Models\Tomador;
use App\Models\TomadorNotaFiscal;
use App\Services\DadosCadastrais\CriaDadosCadastrais;
use App\Services\Prestador\CriaPrestador;
use App\Services\Tomador\BuscaCriaTomador;
use App\Services\Tomador\CriaTomador;
use App\Services\Tomador\CriaTomadorNotaFiscal;
use Carbon\Carbon;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TestCase;

class TomadorTest extends TestCase
{
    use RefreshDatabase;

    private Lote $lote;
    private Rps $rps1;
    private DadosCadastrais $prestadorDadosCadastrais;
    private Instituicao $instituicao;
    private Prestador $prestador;
    private DadosCadastrais $tomadorDadosCadastrais;
    private Tomador $tomador;
    private Cidade $cidade;

    public function test_se_objetos_foram_criados()
    {
        $this->lote->load('rps');

        $this->assertInstanceOf(Lote::class, $this->lote);
        $this->assertInstanceOf(DadosCadastrais::class, $this->prestadorDadosCadastrais);
        $this->assertInstanceOf(DadosCadastrais::class, $this->tomadorDadosCadastrais);
        $this->assertInstanceOf(Instituicao::class, $this->instituicao);
        $this->assertInstanceOf(Prestador::class, $this->prestador);
        $this->assertInstanceOf(Tomador::class, $this->tomador);
        $this->assertInstanceOf(Rps::class, $this->rps1);

        $this->assertCount(1, $this->lote->rps);
    }

    public function test_busca_tomador_quando_ja_cadastrado()
    {
        $tomador = (new BuscaCriaTomador())->buscar(
            $this->rps1,
            $this->prestador->id,
            $this->instituicao->id
        );

        $this->assertIsArray($tomador);
        $this->assertCount(2, $tomador);

        $this->assertEquals(1, $tomador['tomador']->id);
        $this->assertEquals(1, $tomador['tomador']->id);
        $this->assertEquals(2, $tomador['tomador']->id_dados_cadastrais);
        $this->assertEquals(1, $tomador['tomador']->id_prestador);
        $this->assertEquals(1, $tomador['tomador']->id_instituicao);

        $this->assertCount(1, Tomador::all());
        $this->assertCount(1, TomadorNotaFiscal::all());
        $this->assertCount(2, DadosCadastrais::all());
    }

    public function test_criar_tomador_quando_nao_cadastrado()
    {
        //forçando a alteração do cnpj do prestador nos dados cadastrais.
        $this->rps1->cpf_cnpj_tomador = '55555';

        $tomador = (new BuscaCriaTomador())->buscar(
            $this->rps1,
            $this->prestador->id,
            $this->instituicao->id
        );

        $this->assertIsArray($tomador);
        $this->assertCount(2, $tomador);

        //novo tomador criado. id = 2
        $this->assertCount(2, Tomador::all());
        $this->assertCount(2, TomadorNotaFiscal::all());
        $this->assertCount(3, DadosCadastrais::all());

        $this->assertEquals(2, $tomador['tomador']->id);
        $this->assertEquals(3, $tomador['tomador']->id_dados_cadastrais);
        $this->assertEquals(1, $tomador['tomador']->id_prestador);
        $this->assertEquals(1, $tomador['tomador']->id_instituicao);

        //segundo tomador_nota_fiscal criado. id = 2 - já existia um cadastrado.
        $this->assertEquals(2, $tomador['tomadorNotaServico']->id);
        $this->assertEquals($this->cidade->id, $tomador['tomadorNotaServico']->id_cidade_tomador);

        $dadosCadastrais = DadosCadastrais::find(3);

        $this->assertEquals('55555', $tomador['tomadorNotaServico']->cnpj_cpf_tomador);
        $this->assertEquals('55555', $dadosCadastrais->cnpj_cpf);
    }

    protected function setUp(): void
    {
        parent::setUp();

        //Insituição
        $this->instituicao = new Instituicao();
        $this->instituicao->nome = 'PREFEITURA MUNICIPAL DE CAJAZEIRAS';
        $this->instituicao->cnpj = '08923971000115';
        $this->instituicao->situacao = 1;
        $this->instituicao->datainclusao = Carbon::now();
        $this->instituicao->data_alteracao = Carbon::now();
        $this->instituicao->email = 'teste@teste';
        $this->instituicao->rua = 'werr';
        $this->instituicao->numero = 'fsf';
        $this->instituicao->bairro = 'sfsf';
        $this->instituicao->cep = 'fsf';
        $this->instituicao->complemento = 'fsff';
        $this->instituicao->telefone = 'fsf';
        $this->instituicao->logomarca = 'fsf';
        $this->instituicao->link_acesso_sistema = '';
        $this->instituicao->multa = 0;
        $this->instituicao->juros = 0;
        $this->instituicao->aliquota = 0;
        $this->instituicao->dia_vencimento = 10;
        $this->instituicao->tipo_boleto = 1;
        $this->instituicao->dia_limite_cancelamento = 11;
        $this->instituicao->id_cidade = 1290;
        $this->instituicao->save();

        //array Rps
        $this->rps['Id']                         = '10';
        $this->rps['NumeroRps']                  = '9999';
        $this->rps['Serie']                      = 'RPS';
        $this->rps['Tipo']                       = 1;
        $this->rps['DataEmissaoRps']             = Carbon::now();
        $this->rps['NaturezaOperacao']           = '0012487';
        $this->rps['RegimeEspecialTributacao']   = 3;
        $this->rps['OptanteSimplesNacional']     = 2;
        $this->rps['IncentivadorCultural']       = 2;
        $this->rps['Competencia']                = '202109';
        $this->rps['Status']                     = 1;

        //Serviços - Valores
        $this->rps['ValorServicos']              = 350.00;
        $this->rps['ValorDeducoes']              = 0.00;
        $this->rps['ValorPis']                   = 0.00;
        $this->rps['ValorCofins']                = 0.00;
        $this->rps['ValorInss']                  = 0.00;
        $this->rps['ValorIr']                    = 0.00;
        $this->rps['ValorCsll']                  = 0.00;
        $this->rps['ValorIssRetido']             = 0.00;
        $this->rps['ValorIss']                   = 17.50;
        $this->rps['OutrasRetencoes']            = 0.00;
        $this->rps['Aliquota']                   = 0.00;
        $this->rps['DescontoIncondicionado']     = 0.00;
        $this->rps['DescontoCondicionado']       = 0.00;

        //Serviços
        $discriminacao = "NS:12487; OS:50765; Placa:PDQ7484; KM:105159; Chassi:9BFZB55HXF8530201; SRIDS - FERRAMENTA DE DIAGNOSTICO IDS; ST - SUBST. TRIBUTARIA PECAS CONF. PROTOCOLO 61/12 E 62/12.Prop.: THIAGO SILVA BARROS";

        $this->rps['ItemListaServico']           = '1401';
        $this->rps['CodigoCnae']                 = '4520001';
        $this->rps['CodigoTributacaoMunicipio']  = '4520001';
        $this->rps['Discriminacao']              = $discriminacao;
        $this->rps['CodigoMunicipioServico']     = '1100205';

        //Tomador
        $this->rps['CpfCnpjTomador'] = '33333';
        $this->rps['RazaoSocialTomador']                = 'THIAGO SILVA BARROS (121322)';
        $this->rps['EnderecoTomador']                   = 'ANTONIO ALVES CONSERVA';
        $this->rps['NumeroEnderecoTomador']             = '1104';
        $this->rps['ComplementoTomador']                = 'CASA';
        $this->rps['BairroTomador']                     = 'NOSSA SRA DO PERPETUO SOCORRO';
        $this->rps['CodigoMunicipioTomador']            = '1100205';
        $this->rps['UfTomador']                         = 'PE';
        $this->rps['CepTomador']                        = '56000000';

        //Contato
        $this->rps['TelefoneTomador']                   = '(83) 98888-5555';
        $this->rps['EmailTomador']                      = 'tomador@teste.com';

        //Lote
        $this->lote = new Lote();
        $this->lote->lote_rps_id = '123';
        $this->lote->numero = 111;
        $this->lote->cnpj = '52001238000135';
        $this->lote->inscricao_municipal = '321654987';
        $this->lote->quantidade_rps = 10;
        $this->lote->status = StatusEnum::NORMAL;
        $this->lote->instituicao_id = $this->instituicao->id;
        $this->lote->save();

        $this->rps1 = new Rps();
        $this->rps1->rps_id = $this->rps['Id'];
        $this->rps1->numero = $this->rps['NumeroRps'];
        $this->rps1->serie = $this->rps['Serie'];
        $this->rps1->tipo = $this->rps['Tipo'];
        $this->rps1->data_emissao = $this->rps['DataEmissaoRps'];
        $this->rps1->natureza_operacao = $this->rps['NaturezaOperacao'];
        $this->rps1->regime_especial_tributacao = $this->rps['RegimeEspecialTributacao'];
        $this->rps1->optante_simples_nacional = $this->rps['OptanteSimplesNacional'];
        $this->rps1->incentivador_cultural = $this->rps['IncentivadorCultural'];
        $this->rps1->competencia = $this->rps['Competencia'];
        $this->rps1->status = $this->rps['Status'];

        $this->rps1->valor_servicos = $this->rps['ValorServicos'];
        $this->rps1->valor_deducoes = $this->rps['ValorDeducoes'];
        $this->rps1->valor_pis = $this->rps['ValorPis'];
        $this->rps1->valor_cofins = $this->rps['ValorCofins'];
        $this->rps1->valor_inss = $this->rps['ValorInss'];
        $this->rps1->valor_ir = $this->rps['ValorIr'];
        $this->rps1->valor_csll = $this->rps['ValorCsll'];
        $this->rps1->valor_iss_retido = $this->rps['ValorIssRetido'];
        $this->rps1->valor_iss = $this->rps['ValorIss'];
        $this->rps1->outras_retencoes = $this->rps['OutrasRetencoes'];
        $this->rps1->aliquota = $this->rps['Aliquota'];
        $this->rps1->desconto_incondicionado = $this->rps['DescontoIncondicionado'];
        $this->rps1->desconto_condicionado = $this->rps['DescontoCondicionado'];

        $this->rps1->item_listaServico = $this->rps['ItemListaServico'];
        $this->rps1->codigo_cnae = $this->rps['CodigoCnae'];
        $this->rps1->codigo_tributacao_municipio = $this->rps['CodigoTributacaoMunicipio'];

        $this->rps1->discriminacao = $this->rps['Discriminacao'];
        $this->rps1->codigo_municipio = $this->rps['CodigoMunicipioServico'];

        $this->rps1->cpf_cnpj_tomador = $this->rps['CpfCnpjTomador'];
        $this->rps1->razao_social_tomador = $this->rps['RazaoSocialTomador'];
        $this->rps1->endereco_tomador = $this->rps['EnderecoTomador'];
        $this->rps1->numero_tomador = $this->rps['NumeroEnderecoTomador'];
        $this->rps1->complemento_tomador = $this->rps['ComplementoTomador'];
        $this->rps1->bairro_tomador = $this->rps['BairroTomador'];
        $this->rps1->codigo_municipal_tomador = $this->rps['CodigoMunicipioTomador'];
        $this->rps1->uF_tomador = $this->rps['UfTomador'];
        $this->rps1->cep_tomador = $this->rps['CepTomador'];
        $this->rps1->telefone_tomador = $this->rps['TelefoneTomador'];
        $this->rps1->email_tomador = $this->rps['EmailTomador'];
        $this->rps1->lote_id = $this->lote->id;
        $this->rps1->save();

        /**
         * Cidade
         */
        $this->cidade = Cidade::create([
            'nome' => 'PORTO VELHO',
            'codigo' => '1100205',
            'id_estado' => 1
        ]);

        /**
         * ---------------------------------------
         * PRESTADOR
         * ---------------------------------------
         */

        $criadorDeDadosCadastrais = new CriaDadosCadastrais(
            '1111',
            'PRESTADOR TESTE WEBSERVICE',
            '16.139.962-3',
            '0020222',
            1058,
            '58900971',
            '',
            'Centro',
            'Rua do Comércio',
            's/n',
            '',
            'testewebservice@test.com',
            '(83) 98877-6655',
            '',
            TipoPessoaTomador::PESSOA_JURIDICA,
            'Manoel Proprietário',
            1,
        );

        $this->prestadorDadosCadastrais = $criadorDeDadosCadastrais->criar();

        $criadorDePrestador = new CriaPrestador(
            'aaaaaaaaaaaa',
            'REPRESENTANTE NOME TESTE',
            '22222',
            $this->prestadorDadosCadastrais->id
        );

        $this->prestador = $criadorDePrestador->criar();

        /**
         * ---------------------------------------
         * TOMADOR
         * ---------------------------------------
         */

        $criadorDeDadosCadastrais = new CriaDadosCadastrais(
            '33333',
            'Manoel TOMADOR',
            '16.139.962-3',
            '0020222',
            1058,
            '58900971',
            '',
            'Centro',
            'Rua do Comércio',
            's/n',
            '',
            'testewebservice@test.com',
            '(83) 98877-6655',
            '',
            TipoPessoaTomador::PESSOA_FISICA,
            'Manoel TOMADOR',
            1,
        );

        $this->tomadorDadosCadastrais = $criadorDeDadosCadastrais->criar();

        $criadorDeTomadores = new CriaTomador(
            $this->tomadorDadosCadastrais->id,
            $this->prestador->id,
            1
        );

        $this->tomador = $criadorDeTomadores->criar();

        /**
         * Tomador Nota Fiscal
         */
        $criadorDeTomadorNota = new CriaTomadorNotaFiscal(
            'TOMADOR_NOTA JÁ CADASTRADO DO SILVA',
            '33333',
            '58900970',
            'Centro',
            'Rua Coronel Juvêncio Carneiro',
            '489',
            '',
            'tomador@teste.com',
            '(83) 3333-4444',
            '',
            "",
            "",
            '',
            TipoPessoaTomador::PESSOA_JURIDICA,
            1058,
            $this->cidade->id
        );

        $tomadorNota = $criadorDeTomadorNota->criar();

    }
}
