<?php

namespace Tests\Feature;

use App\Enums\SituacaoEnum;
use App\Enums\TipoPessoaTomador;
use App\Models\AtividadeEconomica;
use App\Models\AtividadeEconomicaClasse;
use App\Models\AtividadeEconomicaSubClasse;
use App\Models\Cidade;
use App\Models\DadosCadastrais;
use App\Models\Instituicao;
use App\Models\NotaFiscalServico;
use App\Models\NumeracaoNota;
use App\Models\Prestador;
use App\Models\PrestadorNotaFiscal;
use App\Models\Tomador;
use App\Models\TomadorNotaFiscal;
use App\Services\AtividadeEconomica\CriaAtividadeEconomica;
use App\Services\DadosCadastrais\CriaDadosCadastrais;
use App\Services\Notas\CriadorNotaFiscalServico;
use App\Services\Notas\NotaPersist;
use App\Services\Prestador\BuscaPrestadorNotaFiscal;
use App\Services\Prestador\CriaPrestador;
use App\Services\Prestador\CriaPrestadorNotaFiscal;
use App\Services\Tomador\CriaTomador;
use App\Services\Tomador\CriaTomadorNotaFiscal;
use Carbon\Carbon;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TestCase;

class PrestadorTest extends TestCase
{
    use RefreshDatabase;

    private Prestador $prestador;
    private DadosCadastrais $dadosCadastrais;
    private Cidade $cidade;
    private Instituicao $instituicao;
    private Tomador $tomador;
    private DadosCadastrais $tomadorDadosCadastrais;
    private TomadorNotaFiscal $tomadorNota;
    private AtividadeEconomicaClasse $atividadeClasse;
    private AtividadeEconomicaSubClasse $atividadeSubClasse;
    private AtividadeEconomica $atividadeEconomicaCriada;

    public function test_instaciacao_dos_objetos()
    {
        $this->assertInstanceOf(Prestador::class, $this->prestador);
        $this->assertInstanceOf(DadosCadastrais::class, $this->dadosCadastrais);
    }

    /**
     * @throws \Exception
     */
    public function test_criar_prestador_nota_se_nao_existir()
    {
        $novoPrestadorNotaCriado = BuscaPrestadorNotaFiscal::buscarOuCriarSeNaoExistir($this->prestador->id);

        $this->assertInstanceOf(PrestadorNotaFiscal::class, $novoPrestadorNotaCriado);
        $this->assertEquals(1, $novoPrestadorNotaCriado->id);
        $this->assertCount(1, PrestadorNotaFiscal::all());
    }

    public function test_buscar_prestador_nota_quando_existir()
    {
        $prestadorNotaJaCadastrado = (new CriaPrestadorNotaFiscal(
            "PRESTADOR NOTA CRIADO ANTERIORMENTE",
            "58000000",
            "JOÃO PESSOA / PARAÍBA",
            "CENTRO",
            "PRINCIPAL",
            "10",
            "APENAS TESTE",
            "email@jacadastrado.com",
            "(83) 99999-9999",
            "12",
            2
        ))->criar();

        /**
         * Dados Para a nota.
         */
        $nota['numero'] = "1000001";
        $nota['codigo_verificacao'] = "ABC123";
        $nota['descricao'] = "SERVIÇOS DE TESTE";
        $nota['numero_substituicao'] = null;
        $nota['numero_processo'] = null;
        $nota['situacao'] = SituacaoEnum::REGULAR;
        $nota['data_emissao'] = Carbon::now();

        $nota['competencia'] = Carbon::now();
        $nota['competencia_mes'] = '2';
        $nota['competencia_ano'] = '2021';

        $nota['valor'] = 10.00;
        $nota['valor_iss'] = 0.00;
        $nota['valor_inss'] = 0.00;
        $nota['base_calculo'] = 0.00;
        $nota['valor_liquido'] = 0.00;
        $nota['deducao_legal'] = 0.00;
        $nota['valor_pis'] = 0.00;
        $nota['valor_irrf'] = 0.00;
        $nota['valor_confins'] = 0.00;
        $nota['valor_csll'] = 0.00;
        $nota['outras_retencoes'] = 0.00;
        $nota['aliquota'] = 0.00;
        $nota['data_cancelamento'] = null;
        $nota['exigibilidade_iss'] = 1;
        $nota['retencao_iss'] = 0.00;

        $nota['regime_tributacao_prestador'] = '3';
        $nota['optante_simples_prestador'] = '1';
        $nota['desconto_condicionado'] = 0.00;
        $nota['desconto_incondicionado'] = 0.00;
        $nota['pais_incidencia'] = 1058; // Brasil Cod

        //IDs
        $nota['id_prestador'] = $this->prestador->id;
        $nota['id_tomador'] = $this->tomador->id;
        $nota['id_tomador_nota'] = $this->tomadorNota->id;

        $nota['id_prestador_nota'] = $prestadorNotaJaCadastrado->id;
        $nota['id_atividade'] = $this->atividadeEconomicaCriada->id;

        $nota['id_instituicao'] = $this->instituicao->id;
        $nota['id_cidade_prestacao'] = $this->cidade->id;
        $nota['id_cidade_incidencia'] = $this->cidade->id;

        $nota['id_dam'] = null;
        $nota['id_dam_avulso_externo'] = null;

        $notaCadastrada = NotaPersist::store($nota);

        $numeracaoNota = NumeracaoNota::create([
            'numero' => $notaCadastrada->numero,
            'id_prestador' => $prestadorNotaJaCadastrado->id
        ]);

        $this->assertInstanceOf(NumeracaoNota::class, $numeracaoNota);
        $this->assertInstanceOf(NotaFiscalServico::class, $notaCadastrada);

        $prestadorNotaJaNoBanco = BuscaPrestadorNotaFiscal::buscarOuCriarSeNaoExistir($prestadorNotaJaCadastrado->id);

        $this->assertInstanceOf(PrestadorNotaFiscal::class, $prestadorNotaJaNoBanco);
        $this->assertEquals(1, $prestadorNotaJaNoBanco->id);
        $this->assertEquals("PRESTADOR NOTA CRIADO ANTERIORMENTE", $prestadorNotaJaNoBanco->razao_nome_prestador);
        $this->assertCount(1, PrestadorNotaFiscal::all());
        $this->assertCount(1, Prestador::all());
    }

    protected function setUp(): void
    {
        parent::setUp();

        $this->cidade = Cidade::create([
            'nome' => 'PORTO VELHO',
            'codigo' => '1100205',
            'id_estado' => 1
        ]);

        //Insituição
        $this->instituicao = new Instituicao();
        $this->instituicao->nome = 'PREFEITURA MUNICIPAL DE CAJAZEIRAS';
        $this->instituicao->cnpj = '08923971000115';
        $this->instituicao->situacao = 1;
        $this->instituicao->datainclusao = Carbon::now();
        $this->instituicao->data_alteracao = Carbon::now();
        $this->instituicao->email = 'teste@teste';
        $this->instituicao->rua = 'werr';
        $this->instituicao->numero = 'fsf';
        $this->instituicao->bairro = 'sfsf';
        $this->instituicao->cep = 'fsf';
        $this->instituicao->complemento = 'fsff';
        $this->instituicao->telefone = 'fsf';
        $this->instituicao->logomarca = 'fsf';
        $this->instituicao->link_acesso_sistema = '';
        $this->instituicao->multa = 0;
        $this->instituicao->juros = 0;
        $this->instituicao->aliquota = 0;
        $this->instituicao->dia_vencimento = 10;
        $this->instituicao->tipo_boleto = 1;
        $this->instituicao->dia_limite_cancelamento = 11;
        $this->instituicao->id_cidade = $this->cidade->id;
        $this->instituicao->save();

        $criadorDeDadosCadastrais = new CriaDadosCadastrais(
            '123',
            'PRESTADOR TESTE WEBSERVICE',
            '16.139.962-3',
            '0020222',
            1058,
            '58900971',
            '',
            'Centro',
            'Rua do Comércio',
            's/n',
            '',
            'testewebservice@test.com',
            '(83) 98877-6655',
            '',
            TipoPessoaTomador::PESSOA_JURIDICA,
            'Manoel Proprietário',
            $this->cidade->id,
        );

        $this->dadosCadastrais = $criadorDeDadosCadastrais->criar();

        $criadorDePrestador = new CriaPrestador(
            '3232126546466',
            'REPRESENTANTE TESTE',
            '456',
            $this->dadosCadastrais->id
        );

        $this->prestador = $criadorDePrestador->criar();

        /**
         * ---------------------------------------
         * TOMADOR
         * ---------------------------------------
         */

        $criadorDeDadosCadastrais = new CriaDadosCadastrais(
            '33333',
            'Manoel TOMADOR',
            '16.139.962-3',
            '0020222',
            1058,
            '58900971',
            '',
            'Centro',
            'Rua do Comércio',
            's/n',
            '',
            'testewebservice@test.com',
            '(83) 98877-6655',
            '',
            TipoPessoaTomador::PESSOA_FISICA,
            'Manoel TOMADOR',
            1,
        );

        $this->tomadorDadosCadastrais = $criadorDeDadosCadastrais->criar();

        $criadorDeTomadores = new CriaTomador(
            $this->tomadorDadosCadastrais->id,
            $this->prestador->id,
            1
        );

        $this->tomador = $criadorDeTomadores->criar();

        /**
         * Tomador Nota Fiscal
         */
        $criadorDeTomadorNota = new CriaTomadorNotaFiscal(
            'TOMADOR_NOTA JÁ CADASTRADO DO SILVA',
            '33333',
            '58900970',
            'Centro',
            'Rua Coronel Juvêncio Carneiro',
            '489',
            '',
            'tomador@teste.com',
            '(83) 3333-4444',
            '',
            "",
            "",
            '',
            TipoPessoaTomador::PESSOA_JURIDICA,
            1058,
            $this->cidade->id
        );

        $this->tomadorNota = $criadorDeTomadorNota->criar();

        $this->atividadeClasse = AtividadeEconomicaClasse::create([
            'codigo' => '45.20-0',
            'descricao' => 'Manutenção e reparação de veículos automotores',
            'secao' => 'G',
            'divisao' => '45',
            'codigo_grupo' => '45.2',
            'id_grupo' => 1,
        ]);

        $this->atividadeSubClasse = AtividadeEconomicaSubClasse::create([
            'subclasse' => '4520-0/01',
            'nome_subclasse' => 'Serviços de manutenção e reparação mecânica de veículos automotores',
            'classe' => '45.20-0',
            'nome_classe' => 'Manutenção e reparação de veículos automotores',
            'grupo' => '45.2',
            'nome_grupo' => 'Manutenção e reparação de veículos automotores',
            'divisao' => '45',
            'nome_divisao' => 'COMÉRCIO E REPARAÇÃO DE VEÍCULOS AUTOMOTORES E MOTOCICLETAS'
        ]);

        $criadorDeAtividadeEconomica = new CriaAtividadeEconomica(
            1,
            $this->prestador->id,
            $this->atividadeClasse->id,
            $this->atividadeSubClasse->id,
            1
        );

        $this->atividadeEconomicaCriada = $criadorDeAtividadeEconomica->criar();

    }
}
