<?php

namespace App\Services\Balancete;

use App\Enums\StatusPadraoEnum;
use App\Models\InstituicaoFinanceira\DataLimteEnvioFinanceiro;
use App\Services\ZerosAEsquedaParaUmMes;
use RuntimeException;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Log;

class VerificadorDePrazoLimeteParaEnvio
{
    private string $data_hoje;
    private string $vencimento_envio_sem_multa;
    private int $dia_limite_envio;
    public function __construct(
        private Model $balancete
    )
    {
        $this->data_hoje = date('Y-m-d');
    }

    /**
     * @return bool
     * @description  A partir do mês/ano do balancete, calcula-se a data limite para o envio do arquivo csv.
     * Exemplo: se o arquivo a ser enviado for do mês de janeiro, e se a prefeitura determinou o dia 10 do mês subsequente com prazo limite,
     * a partir do dia 11/02 já será considerado fora de prazo.
     * Ou seja, se a data de hoje for menor ou igual ao cálculo, retorna true; Senão, retorna false.
     */
    public function balanceteEstaNoPrazo() : bool
    {
        try {

            $this->obterPrazoParaMulta();

            $mes_competencia_balancete = ZerosAEsquedaParaUmMes::get($this->balancete->mes_competencia);

            $calculo_data_limite_para_envio_do_balancete = date('Y-m-d',
                mktime(0,
                    0,
                    0,
                    (int) $mes_competencia_balancete + 1,
                    $this->dia_limite_envio,
                    $this->balancete->ano_competencia
                )
            );

            $this->vencimento_envio_sem_multa = $calculo_data_limite_para_envio_do_balancete;

            return strtotime($this->data_hoje) <= strtotime($calculo_data_limite_para_envio_do_balancete);

        } catch (RuntimeException $exception) {
            Log::alert("(VerificadorDePrazoLimeteParaEnvio) : {$exception->getMessage()}");
            throw new RuntimeException($exception->getMessage());
        }
    }

    public function vencimento(): string
    {
        $this->balanceteEstaNoPrazo();
        return $this->vencimento_envio_sem_multa;
    }

//    private function ajustarZerosAEsquedaParaMeses(string $mes): string
//    {
//        return strlen($mes) === 1
//            ? str_pad( $mes, 2 , '0' , STR_PAD_LEFT)
//            : $mes;
//    }

    /**
     * @return void
     * @throws RuntimeException
     */
    private function obterPrazoParaMulta()
    {
        try {

            $instituicao = $this->balancete->prestador->instituicao;

            $prazo = $this->buscarInformacoesParametrizacaoMulta($instituicao);

            if (!$prazo) {
                throw new RuntimeException("A {$instituicao->nome}  - CNPJ {$instituicao->cnpj}, ainda não estabeleceu um prazo limite para o envio dos balancetes.");
            }

            $this->dia_limite_envio = $prazo->dia_limite_envio;

        } catch (RuntimeException $exception) {
            throw new RuntimeException("(obterPrazoParaMulta) - ". $exception->getMessage());
        }
    }

    /**
     * @param $instituicao
     * @return mixed
     */
    public function buscarInformacoesParametrizacaoMulta($instituicao): mixed
    {
        try {
            return DataLimteEnvioFinanceiro::where('instituicao_id', $instituicao->id)
                ->where('status', StatusPadraoEnum::ATIVO)
                ->first();
        } catch (RuntimeException $exception) {
            throw new RuntimeException("(buscarInformacoesParametrizacaoMulta) - ". $exception->getMessage());
        }
    }
}
