<?php

namespace App\Http\Livewire\BalanceteFinanceiro;

use App\Enums\AplicaMultaBalanceteEnum;
use App\Enums\MesEnum;
use App\Imports\CSV\ImportCSV;
use App\Models\InstituicaoFinanceira\BalancetePrestador;
use App\Services\Balancete\VerificadorDePrazoLimeteParaEnvio;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Livewire\Component;
use Livewire\WithFileUploads;
use Exception;

class UploadCsvBalanceteModal extends Component
{
    use WithFileUploads;

    public $showForm;
    public $alert;
    public $arquivo_csv;
    public $uuid;
    public $balancete;
    public $competencia_para_modal;
    public $mensagem_fora_de_prazo_modal;
    protected $geranciadorDePrazoParaEnvio;
    public $envioEstaNoPrazo;
    public $dia_limite_envio;
    public $valor_multa;
    public $instituicao_nome;

    protected $listeners = ['abrirUploadCsvBalanceteModal'];

    protected $rules = [
        'arquivo_csv'  => 'mimes:csv,xls,xlsx',
    ];

    public function render()
    {
        return view('livewire.balancete-financeiro.upload-csv-balancete-modal');
    }

    public function abrirUploadCsvBalanceteModal(string $balancete_uuid): void
    {
        $this->showForm = true;
        $this->alert = [];
        $this->arquivo_csv = null;
        $this->mensagem_fora_de_prazo_modal = null;
        $this->uuid = $balancete_uuid;

        $this->balancete = BalancetePrestador::with('prestador')
            ->where('uuid', $this->uuid)
            ->first();

        $this->competencia_para_modal = MesEnum::from($this->balancete->mes_competencia)->name ."/".$this->balancete->ano_competencia;

        $this->geranciadorDePrazoParaEnvio = new VerificadorDePrazoLimeteParaEnvio($this->balancete);
        $this->envioEstaNoPrazo = $this->geranciadorDePrazoParaEnvio->balanceteEstaNoPrazo();

        $instituicao = $this->balancete->prestador->instituicao;
        $this->instituicao_nome = $instituicao->nome;

        $dadosPrazoLimite = $this->geranciadorDePrazoParaEnvio->buscarInformacoesParametrizacaoMulta($instituicao);
        $this->dia_limite_envio = $dadosPrazoLimite->dia_limite_envio;
        $this->valor_multa = $dadosPrazoLimite->valor_multa;

        if (!$this->envioEstaNoPrazo) {
           $this->exibirMensagemForaDoPrazo();
        }
    }

    /**
     * @throws Exception
     */
    public function submit(): void
    {
        $this->validate();

        try {

            sleep(1);

            if (is_null($this->balancete)) {
                throw new Exception("UploadCsvBalanceteModal: Balancete é null.");
            }

            $folder_storage = "balancetes/balancete_".$this->balancete->id . '_' . $this->balancete->prestador->id;
            $path_stored = $this->arquivo_csv->store($folder_storage);

            $qtdeLancamentosImportados = (new ImportCSV($path_stored, $this->balancete))->loadCsvPath();

            if ($qtdeLancamentosImportados === 0) {

                $this->excluirCsvDoServidor($path_stored);

                throw new Exception("Nenhum lançamento salvo para o Balancete "
                    . MesEnum::from($this->balancete->mes_competencia)->name ."/".$this->balancete->ano_competencia);
            }

            $this->balancete->path_arquivo_csv = $path_stored;
            $this->balancete->aplica_multa = !$this->envioEstaNoPrazo ? AplicaMultaBalanceteEnum::SIM->value : AplicaMultaBalanceteEnum::NAO->value;
            $this->balancete->save();

            $this->showForm = false;

            $this->alert = [
                'show' => true,
                'type' => 'success',
                'title' => 'Sucesso!',
                'description' =>  "Importação de lançamentos para o Balancete de competência "
                    . MesEnum::from($this->balancete->mes_competencia)->name ."/{$this->balancete->ano_competencia}, concluída. <br>"
                    . $qtdeLancamentosImportados . " lançamentos importados com sucesso."
            ];

            $this->emit('refreshDatatable');

        } catch (Exception $exception) {

            Log::error("UploadCsvBalanceteModal: " .$exception->getMessage());

            $this->alert = [
                'show' => true,
                'type' => 'danger',
                'title' => 'Erro!',
                'description' => "Houve um problema durante o procedimento de envio do arquivo CSV/XSL: para o balancete de competência "
                    . MesEnum::from($this->balancete->mes_competencia)->name ."/". $this->balancete->ano_competencia
            ];
        }
    }

    private function excluirCsvDoServidor(string $csv_path) : void
    {
        if (Storage::disk('public')->exists($csv_path)) {
            Storage::disk('public')->delete($csv_path);
        }
    }

    private function exibirMensagemForaDoPrazo()
    {
        $msn = "<strong>Atenção!</strong> <br> A data estabelecida para o envio do balancete sem multa é <strong> até o dia ";
        $msn .= $this->dia_limite_envio. " de cada mês subsequente ao mês da competência.</strong> ";
        $msn .= "Para os balancetes enviados após a data limite, <strong>será acrescida multa de ";
        $msn .= $this->valor_multa. "% ao valor do DAM</strong> (Documento de Arrecadação Municipal).";

        $this->mensagem_fora_de_prazo_modal = $msn;
    }
}
